//  File: reminder.cc
// 
//      This file is part of minkowsky
// 
//      Copyright (C) 2001-2002 by Rdiger Goetz
//      Author: Rdiger Goetz <minkowsky@r-goetz.de>
// 
//      Time-stamp: <28-May-2002 12:52:08 goetz>
// 
//      This program is free software; you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation; either version 2 of the License, or
//      (at your option) any later version.
// 
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.
// 
//      You should have received a copy of the GNU General Public License
//      along with this program; if not, write to the Free Software
//      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//  
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>  
#include <time.h>    

#include "termin.h"

void extendReminderList();
int add2RemindeList (Reminder *r);
int calcDateBackward (int datum, int tim, int offset , int offset2, int unit);
int calcDateForward (int datum, int tim, int offset , int offset2, int unit);
int sendReminder (Reminder *r);
int sendreminder2User(Reminder *r,int id);
int sendreminder2Group(Reminder *r,int id);
int sendReminderByEmail(Reminder *r,int uid);
int sendReminderByDialog(Reminder *r,int uid);
int sendReminderBySignal(Reminder *r,int uid);
int compareReminder(Reminder *r1, Reminder *r2);
int compReminder(const void * p1,const void *p2);
int recalcRepeatReminder(Date *d,Reminder *r);
int str2Reminder(Reminder *r,char *t,int startDate, int startTime ,int reminder4Date);
char *reminder2Str(Reminder *r,char *t,int reminder4Str);
int ignoreReminder(int did,int uid);
char * formatMail(char *mask,char *text,Reminder *r,int ext);
void readReminderTemplate(char *target,char *txt);

FILE *fprl;
void  getDefaultMailTemplatesFromFile();
void getDefaultMailTemplates_de ();
void getDefaultMailTemplates_en ();

void initReminderList()
{
 int i;
 char file[1024];

 reminderList.list = new ( Reminder *) [DATEBLOCKSIZE];
 reminderList.anz =0;
 reminderList.max =DATEBLOCKSIZE;
 reminderList.blocks =1;
 for(i=0;i<DATEBLOCKSIZE; i++)
  reminderList.list[i]=NULL;
 sprintf(file,"%s/reminder.log",dataDir);
 fprl=fopen(file,"a");
 if(fprl !=NULL)
   {
   time_t tim;
   tim=time(NULL);
   fprintf(fprl,"\n==============================================================================\n\n%s: Minkowsky restarted\n******************************************************************************\n",strChop(ctime(&tim)));
   fflush(fprl);
  }
 else
  {
   if (verbose)
    printf("Can't open 'reminder.log\n");
  }
 switch (serverLang)
  {
  case MINKO_LANG_DE:
   getDefaultMailTemplates_de();
   break;
  case MINKO_LANG_EN:
  default:
   getDefaultMailTemplates_en();
  }
 getDefaultMailTemplatesFromFile();
}
void  getDefaultMailTemplatesFromFile()
{
 char file[1024];
 FILE *fp;

 readReminderTemplate("EinladungExtern.text",       mailMaskEinlExternText);
 readReminderTemplate("EinladungExtern.subject",    mailMaskEinlExternSubj);
 readReminderTemplate("EinladungIntern.text",       mailMaskEinlInternText);
 readReminderTemplate("EinladungIntern.subject",    mailMaskEinlInternSubj);

 readReminderTemplate("ErinnerungExtern.text",      mailMaskErinExternText);
 readReminderTemplate("ErinnerungExtern.subject",   mailMaskErinExternSubj);
 readReminderTemplate("ErinnerungIntern.text",      mailMaskErinInternText);
 readReminderTemplate("ErinnerungIntern.subject",   mailMaskErinInternSubj);

 readReminderTemplate("ErinnerungToDO.text",        mailMaskErinToDoText);
 readReminderTemplate("ErinnerungToDO.subject",     mailMaskErinToDoSubj);
 readReminderTemplate("MahnungToDO.text",           mailMaskMahnToDoText);
 readReminderTemplate("MahnungToDO.subject",        mailMaskMahnToDoSubj);

 readReminderTemplate("ShiftDateIntern.text",       mailMaskShiftInternText);
 readReminderTemplate("ShiftDateIntern.subject",    mailMaskShiftInternSubj);
 readReminderTemplate("ShiftDateExtern.text",       mailMaskShiftExternText);
 readReminderTemplate("ShiftDateExtern.subject",    mailMaskShiftExternSubj);

 readReminderTemplate("SuspendDateIntern.text",     mailMaskSuspInternText);
 readReminderTemplate("SuspendDateIntern.subject",  mailMaskSuspInternSubj);
 readReminderTemplate("SuspendDateExtern.text",     mailMaskSuspExternText);
 readReminderTemplate("SuspendDateExtern.subject",  mailMaskSuspExternSubj);
}

void getDefaultMailTemplates_de ()
{

 mailMaskEinlExternText="Einladung\n\n%E von %A %l zu folgendem Termin ein:\n\n\t%T\n\nBeginn: %B\nEnde  : %S\nOrt   : %s\n\n\nweitere Informationen\n\n%D\n\n--\nDies ist eine automatische generierte Einladung der %M bei \n\t%A\n";
 mailMaskEinlExternSubj="Einladung von %A";
 mailMaskEinlInternText="Einladung\n\n%E %l zu folgendem Termin ein:\n\n\t%T\n\nBeginn: %B\nEnde  : %S\nOrt   : %R\n\n\nweitere Informationen\n\n%D\n\nDer Termin wurden von %I eingetragen.\n\n--\nDies ist eine automatische generierte Einladung der %M bei \n\t%A\n";
 mailMaskEinlInternSubj="Einladung von %A";

 mailMaskErinExternText="Erinnerung\n\nFolgender Termin wurde von %E bei %A angesetzt:\n\n\t%T\n\nBeginn: %B\nEnde  : %S\nOrt   : %R\n\n\nweitere Infromationen\n\n%D\n\n\n--\nDies ist eine automatische Erinnerung der %M bei \n\t%A\n";
 mailMaskErinExternSubj="Terminerinnerung: %T";
 mailMaskErinInternText="Erinnerung\n\nFolgender Termin wurde von %E angesetzt:\n\n\t%T\n\nBeginn: %B\nEnde  : %S\nOrt   : %R\n\n\nweitere Infromationen\n\n%D\n\n\n--\nDies ist eine automatische Erinnerung der %M bei \n\t%A\n";
 mailMaskErinInternSubj="Terminerinnerung: %T";
 
 mailMaskErinToDoText="Erinnerung\n\nDie Aufgabe \n\n\t %T\n\nsoll bis %B fertiggestellt sein.\n\n\nWeitere Informationen\nInitiator:       %I\nAufgabendetails: %D\n\n\n--\nDies ist eine automatische Erinnerung der %M bei \n\t%A\n";
 mailMaskErinToDoSubj="Erinnerung: Aufgabe '%T' fertigstellen";
 mailMaskMahnToDoText="MAHNUNG\n\nDie Fertigstellung der Aufgabe \n\n\t %T\n\nist berfllig\nDer Zeiltermin war %S\n\n\nWeitere Informationen\nInitiator:       %I\nAufgabendetails: %D\n\n\n--\nDies ist eine automatische Mahnung der %M bei \n\t%A\n";
 mailMaskMahnToDoSubj="Mahnung: Aufgabe '%s' ist versptet";
 
 mailMaskShiftInternText="Terminverlegung:\n Der Termin %T wurde auf \n%B \nnach %R verlegt\n\n%?NBegrndung:\n %N \n%]\nDies ist eine automatische Miteilung der %M bei \n\t%A\n";
 mailMaskShiftInternSubj="Terminverlegung: %T";
 mailMaskShiftExternText="Terminverlegung:\n Der Termin %T wurde auf \n%B \nnach %R verlegt\n\n%?NBegrndung:\n %N \n%]\nDies ist eine automatische Miteilung der %M bei \n\t%A\n";
 mailMaskShiftExternSubj="Der Termin '%T' wurde abgesagt";

 mailMaskSuspInternText="Terminabsage:\n Der regelmsige Termin:\n\t %T\n wurde am %B abgesagt\n%?NBegrndung:\n %N\n%]\nDies ist eine automatische Miteilung der %M bei \n\t%A\n";
 mailMaskSuspInternSubj="'%T' abgesagt";
 mailMaskSuspExternText="Terminabsage:\n Der regelmsige Termin:\n\t %T\n wurde am %B abgesagt\n%?NBegrndung:\n %N\n%]\nDies ist eine automatische Miteilung der %M bei \n\t%A\n";
 mailMaskSuspExternSubj="Der Termin %T wurde abgesagt";
}

void getDefaultMailTemplates_en ()
{
 //FIXME translations needed
 getDefaultMailTemplates_de ();
}


void readReminderTemplate(char *target,char *txt)
{
 int fd;
 struct stat  stat;
 char file[1024];
 FILE *fp;

 sprintf(file,"%s/templates/%s",dataDir,target);
 fp = fopen (file,"r");
 if (fp != NULL)
  {
   fd = fileno(fp);
   fstat(fd,& stat);
   if(txt !=NULL)
    {
     delete txt;
     txt=NULL;
    }
   txt = new char [ stat.st_size +16];
   fread(txt,sizeof(char), stat.st_size+1,fp);
   txt[ stat.st_size]='\0';
   fclose(fp);
  }
}
void clearReminderList ()
{
 delete reminderList.list;
}

void extendReminderList()
{
 Reminder **tmp;
 int anz,anz0;
 int i;
 
 reminderList.blocks++;
 anz= dateBlocks * DATEBLOCKSIZE;
 tmp= new ( Reminder *) [anz];
 reminderList.max=anz;
 anz0= anz - DATEBLOCKSIZE;

 memcpy(tmp,reminderList.list,anz0*sizeof(Reminder *));
 delete reminderList.list;
 reminderList.list=tmp;
 for(i=anz0;i<anz; i++)
  reminderList.list[i]=NULL;
}

char *reminder2Str4Todo(Reminder *r,char *t)
{
 return(reminder2Str(r,t,false));
}
char *reminder2Str4Date(Reminder *r,char *t)
{
 return(reminder2Str(r,t,true));
}
char *reminder2Str(Reminder *r,char *t,int reminder4Str)
{
 char *us,*tp,name[32],full[128],typ;
 float off;

 us="?";
 tp="?";

 // Note translations of time interval and destination from  German to englisch are done
 // in the client
 switch (r->unit)
  {
  case 'm' :
   us = "Minuten";
   off= r->offset;
   break;
  case 'h' :
   us = "Stunden";
   off= r->offset + r->offset2/60.;
   break;
  case 'd' :
   us="Tage";
   off= r->offset + r->offset2/24.;
   break;
  }
 switch( (r->typ & REMINDER_DEST_MASK) )
  {
  case REMINDER_BY_EMAIL:
   tp="E-mail";
   break;
  case REMINDER_BY_DIALOG:
   tp="Fenster";
   break;
  case REMINDER_BY_SIGNAL:
   tp="Signal";
   break;
  }
 if(r->id<GROUPOFFSET)
  {
   strcpy(name,userList[r->id].name);
   strcpy(full,userList[r->id].full);
  } 
 else if (r->id<ROOMOFFSET)
  {
   strcpy(name,groupList[r->id-GROUPOFFSET].name);
   sprintf(full,"Gruppe %s",groupList[r->id-GROUPOFFSET].name);
  } 
 else if (r->id>=ADDRESOFFSET && r->id <= MAX_ID)
  {
   int idx;
   aid2cname(r->id-ADDRESOFFSET,full);
   strcpy(name,full);
  } 
 else if(r->id == REMIND_ALL_USER)
  {
   strcpy(name,"alle");
   sprintf(full,"Alle");
  }
 else if(r->id == REMIND_ALL_INTERN)
  {
   strcpy(name,"intern");
   sprintf(full,"interne Teilnehmer");
  }
 else if(r->id == REMIND_ALL_EXTERN)
  {
   strcpy(name,"extern");
   sprintf(full,"externe Teilnehmer");
  }

 switch (r->typ & REMINDER_TYPE_MASK)
  {
  case REMINDER_EINLADUNG:  typ='E'; break;
  case REMINDER_ERINNERUNG: typ='R'; break;
  case REMINDER_MAHNUNG:    typ='M'; break;
  default:                  typ='R'; break;
  }
 sprintf(t,"{%5.2f %s %s %s {%s} %c} ",off,us,tp,name,full,typ);
 return(t);
}


int addReminder2Date ( Date *d, Reminder *r)
{
 if((r->typ & REMINDER_MAHNUNG)>0)
  r->ct.ctrl->tim = calcDateForward(d->startDate,d->startTime,r->offset,r->offset2,r->unit);
 else
  r->ct.ctrl->tim = calcDateBackward(d->startDate,d->startTime,r->offset,r->offset2,r->unit);
 if((r->ct.ctrl->tim - time(NULL)) > -60 )
  r->status = REMINDER_WAITING;
 else 
  r->status = REMINDER_DONE;
 memcpy(&d->reminder[d->remAnz],r,sizeof(Reminder));
 d->remAnz++;
 add2RemindeList(r);
 return(true);
}

int addReminder2Date ( Date *d, char *t)
{
 Reminder *r;

 if( strlen(trimRight(t))==0)
  return(true);

 r= & d->reminder[d->remAnz];
 if(!str2Reminder(r,t,d))
  return(false);
 d->remAnz++;
 add2RemindeList(r);
 return(true);
}
int sendReminder4DateShift ( DateShift *dsh,int refDate,int refTime,int target)
{
 Reminder *r;
 Date *d;

 if(dsh->did<0)
  return false;
 r = &dsh->reminder;

 r->id      = target;
 r->status  = REMINDER_WAITING;
 r->offset  =-1;
 r->offset2 =-1;
 r->unit    =' ';
 r->typ     = REMINDER_SHIFT;
 r->ct.ctrl = new ReminderControl;
 r->date    = (void *)dsh;

 r->ct.ctrl->tim = time(NULL);  // DateShift_reminder werden sofort  verschickt
 r->ct.ctrl->startDate = refDate;
 r->ct.ctrl->startTime = refTime;
 sendReminder(r);
 // add2RemindeList(r);
 return(true);
}

int addReminder2DateShift ( Date *d, char *t)
{
 Reminder *r;

 if( strlen(trimRight(t))==0)
  return(true);

 r= & d->reminder[d->remAnz];
 if(!str2Reminder(r,t,d))
  return(false);
 d->remAnz++;
 add2RemindeList(r);
 return(true);
}

int buildTempReminderList (const Date *d, Reminder *r,char *t)
{
 int anz =0,ranz=0,i;
 char *tp[256];

 if( strlen(trimRight(t))==0)
  return(true);

 for(i=0;i<REMINDERS_IN_DATE; i++)
  r[i].ct.ctrl = NULL;

 anz=split(t,tp,255);

 for(i=0;i<anz; i++)
  {
   if(str2Reminder(&r[ranz],tp[i],d))
    {
     delete r[ranz].ct.ctrl;
     r[ranz].ct.ctrl =NULL;
     ranz++;
    }
  }
 return(ranz);
}
int addReminder2Date ( TodoItem *todo, Reminder *r)
{
 if((r->typ & REMINDER_MAHNUNG)>0)
  r->ct.ctrl->tim = calcDateForward(todo->startDate,1800,r->offset,r->offset2,r->unit);
 else
  r->ct.ctrl->tim = calcDateBackward(todo->startDate,1800,r->offset,r->offset2,r->unit);
 if((r->ct.ctrl->tim - time(NULL)) > -60 )
  r->status = REMINDER_WAITING;
 else 
  r->status = REMINDER_DONE;
 memcpy(&todo->reminder[todo->remAnz],r,sizeof(Reminder));
 todo->remAnz++;
 add2RemindeList(r);
 return(true);
}
int addReminder2Todo ( TodoItem *todo, char *t)
{
 Reminder *r;

 if( strlen(trimRight(t))==0)
  return(true);

 r= & todo->reminder[todo->remAnz];
 if(!str2Reminder(r,t,todo))
  return(false);
 todo->remAnz++;
 add2RemindeList(r);
 return(true);
}
int str2Reminder(Reminder *r,char *t,const TodoItem *todo)
{
 if(!str2Reminder(r,t,todo->startDate,1000,false))
  return(false);
 r->date = (void *) todo;
 return(true);
}
int str2Reminder(Reminder *r,char *t,const Date *d)
{
 if(!str2Reminder(r,t,d->startDate,d->startTime,true))
  return(false);
 printf("got reminder for Date %s : (%d/%d) - %d;%d %c => %d/%s (%p)\n",
	d->descr,d->startDate,d->startTime,r->offset,r->offset2,r->unit,r->ct.ctrl->tim,strTrim(ctime((const time_t*) &r->ct.ctrl->tim)), r);
 r->date = (void *) d;
 return(true);
}
int str2Reminder(Reminder *r,char *t,int startDate, int startTime ,int reminder4Date)
{
 char *tp[8];
 float offset;
 int id;

 split(t,tp,8);

 if(r->ct.ctrl== NULL)
  r->ct.ctrl = new ReminderControl ;


 r->typ =0;
 if( strstr(tp[2],"E-mail") != NULL)
  r->typ |= REMINDER_BY_EMAIL;
 if( strstr(tp[2],"Fenster") != NULL)
  r->typ |= REMINDER_BY_DIALOG;
 if( strstr(tp[2],"Signal") != NULL)
  r->typ |= REMINDER_BY_SIGNAL;

 if(!reminder4Date)
  r->typ != REMINDER_FOR_TODO;

 if(tp[5][0] == 'E')
  r->typ |= REMINDER_EINLADUNG;
 if(tp[5][0] == 'R')
  r->typ |= REMINDER_ERINNERUNG;
 if(tp[5][0] == 'M')
  r->typ |= REMINDER_MAHNUNG;

 if(r->typ ==0)
  printf("REMINDER: strange Typ %d / %s\n",r->typ,tp[2]);

 offset = atof(tp[0]);
 if(strcmp(tp[1],"Tage")==0)
  {
   r->unit     = 'd';
   r->offset   = (int)floor(offset);
   r->offset2  = (int)floor((offset- r->offset)*24);
  }
 else if(strcmp(tp[1],"Stunden")==0)
  {
   r->unit     = 'h';
   r->offset   = (int)floor(offset);
   r->offset2  = (int)floor((offset- r->offset)*60);
  }
 else 
  {
   r->unit     = 'm';
   r->offset   = (int)floor(offset);
   r->offset2  = 0;
  }
 if((r->typ & REMINDER_MAHNUNG)>0)
  r->ct.ctrl->tim = calcDateForward(startDate,startTime,r->offset,r->offset2,r->unit);
 else
  r->ct.ctrl->tim = calcDateBackward(startDate,startTime,r->offset,r->offset2,r->unit);
 if((r->ct.ctrl->tim - time(NULL)) > -60 )
  {
   if(r->ct.ctrl->tim != r->ct.ctrl->lastSendTim)
    r->status = REMINDER_WAITING;
   else
    r->status = REMINDER_DONE;
  }
 else 
  r->status = REMINDER_DONE;
 

 
 if(strcmp(tp[3],"alle")==0)
  id =REMIND_ALL_USER;
 else if (strcmp(tp[3],"intern")==0)
  id =REMIND_ALL_INTERN;
 else if (strcmp(tp[3],"extern")==0)
  id =REMIND_ALL_EXTERN;
 else
  {
   id=uname2uid(tp[3]);
   if(id<0)
    {
     id = gname2gid(tp[3]);
     if(id>=0)
      id += GROUPOFFSET;
     if(id<0)
      {
       id= aname2aid(tp[3]);
       if(id>=0)
	id+= ADDRESOFFSET;
      }
    }
  }

 if(id<0)
  return(false);
 r->id=id;
 return(true);
}

int recalcReminder(Date *d)
{
 int i;
 Reminder *r;


 for(i=0;i< d->remAnz; i++)
  {
   r = & d->reminder[i];

   if(r->ct.ctrl ==NULL)
    {
     r->ct.ctrl = new ReminderControl;
     r->ct.ctrl->lastSendTim= 0;
    }
   
   if((r->typ & REMINDER_MAHNUNG)>0)
    r->ct.ctrl->tim = calcDateForward(d->startDate,d->startTime,r->offset,r->offset2,r->unit);
   else
    r->ct.ctrl->tim = calcDateBackward(d->startDate,d->startTime,r->offset,r->offset2,r->unit);
   
//    if(verbose)
//     printf("\tReminder %s;%d: %d/%s // %p; %p\n",
// 	   d->descr,i,r->ct.ctrl->tim,strTrim(ctime((const time_t*) &r->ct.ctrl->tim)),r->ct.ctrl,&r->ct.ctrl->tim);

   if(r->ct.ctrl->lastSendTim < r->ct.ctrl->tim)
    r->status = REMINDER_WAITING;
   // FIXME dateShift
   r->ct.ctrl->startDate = d->startDate;
   r->ct.ctrl->startTime = d->startTime;
   r->ct.ctrl->stopDate  = d->stopDate;
   r->ct.ctrl->stopTime  = d->stopTime;
  }
//  if(verbose && d->remAnz>0)
//   {
//    printf("summary for %s\n",d->descr);
//    for(i=0;i< d->remAnz; i++)
//      printf("%3d: %d/%s %p %p\n",i,d->reminder[i].ct.ctrl->tim, strTrim(ctime((const time_t*) &d->reminder[i].ct.ctrl->tim)), d->reminder[i].ct.ctrl,&d->reminder[i].ct.ctrl->tim);
//    printf("------\n\n");
//   }
 return(true);
}

int resetReminderStates(Date *d)
{
 int i;
 for(i=0;i< d->remAnz; i++)
  d->reminder[i].status = REMINDER_WAITING;
}
int recalcRepeatReminder(Date *d,Reminder *r)
{
 int i,rtim;
 CalendarItem ci;


 if(r->ct.ctrl ==NULL)
  {
   r->ct.ctrl = new ReminderControl;
   r->ct.ctrl->tim =0;
  }

 rtim = r->ct.ctrl->tim;
 nextDateAfterNow(&ci,d);
 r->ct.ctrl->startDate = ci.startDate;
 r->ct.ctrl->startTime = ci.startTime;
 r->ct.ctrl->stopDate  = ci.stopDate;
 r->ct.ctrl->stopTime  = ci.stopTime;
 if((r->typ & REMINDER_MAHNUNG)>0)
  r->ct.ctrl->tim = calcDateForward(ci.startDate,ci.startTime,r->offset,r->offset2,r->unit);
 else
  r->ct.ctrl->tim = calcDateBackward(ci.startDate,ci.startTime,r->offset,r->offset2,r->unit);
 if(r->ct.ctrl->tim > r->ct.ctrl->lastSendTim)
  r->status = REMINDER_WAITING;

 return(true);
}


int findReminderInList(Reminder *r, Reminder *list, int anz)
{
 int i;
 int identical;
 for(i=0;i<anz; i++)
  {
   if(compareReminder(r,&list[i]))
    return (i);
  }
 return(-1);
}

int compareReminder(Reminder *r1, Reminder *r2)
{
 if(r1->id      != r2->id     )
  return(false);
 if(r1->offset  != r2->offset )
  return(false);
 if(r1->offset2 != r2->offset2)
  return(false);
 if(r1->unit    != r2->unit   )
  return(false);
 if(r1->typ     != r2->typ    )
  return(false);

 return (true);
}

int recreateReminderList()
{
 int i,r,ctim;
 int offset;

 ctim   = time(NULL) ;
 ctim  -= (ctim %60);

 //FIXME  Recreate f[r TODO-Reminder fehlt

 if(verbose)
  printf("REbuilding reminderList\n");
 reminderList.anz =0;
 for (i=0;i<dateAnz;i++)
  if(dateList[i] !=NULL)
   if(!dateList[i]->deleted)
    {
//      if(verbose)
//       if(dateList[i]->remAnz>0)
//        printf("found %d reminder in date %d/%s\n",dateList[i]->remAnz,i,dateList[i]->descr);
     for(r=0;r<dateList[i]->remAnz; r++)
      {
       switch(dateList[i]->reminder[r].unit)
	{
	case 'm': offset =      1800+30; break;
	case 'h': offset =      3600+30; break;
	case 'd': offset =    4*3600+30; break;
	default:  offset = 30; break;
	}
       if(dateList[i]->repeatTyp != NO_REPEAT)
	recalcRepeatReminder(dateList[i],&dateList[i]->reminder[r]);
//        printf("rtim =%d   ctim=%d  delta=%d // typ %d \n",dateList[i]->reminder[r].ct.ctrl->tim,ctim,dateList[i]->reminder[r].ct.ctrl->tim - ctim,dateList[i]->repeatTyp);
//        printf("\t %s \n",ctime((time_t*)&dateList[i]->reminder[r].ct.ctrl->tim));
//        printf("\t vs %s\n",ctime((time_t*)&ctim));
       if( dateList[i]->reminder[r].ct.ctrl->tim - ctim > -offset)
	{
	 if(verbose)
	  {
	   char now[256];
	   strcpy(now,strTrim(ctime((const time_t*)&ctim)));
	   printf("\t checking %d: %d; rem on %10d/%s;\n\t\t\tnow:   %10d/%s\n",r,dateList[i]->reminder[r].status,
		  dateList[i]->reminder[r].ct.ctrl->tim,strTrim(ctime((const time_t*)&dateList[i]->reminder[r].ct.ctrl->tim)),
		  ctim,now);
	  }
	 if(verbose)
	  printf("\t\t     lastSend: %10d/%s\n",dateList[i]->reminder[r].ct.ctrl->lastSendTim,strTrim(ctime((const time_t*)&dateList[i]->reminder[r].ct.ctrl->lastSendTim)));
	 if(dateList[i]->reminder[r].status == REMINDER_WAITING || dateList[i]->repeatTyp != NO_REPEAT)
	  {
	   if(verbose)
	    printf("\t\t adding reminder %d on position %d\n",r,reminderList.anz);
	   add2RemindeList(&dateList[i]->reminder[r]);
	   dateList[i]->reminder[r].date = dateList[i];
	  }
	}
      }
    }

 updateReminderList();
 nextReminder =0;
 lastReminderListCreation =time(NULL);
 for(r=nextReminder; r<reminderList.anz; r++)
  if(reminderList.list[r]->status == REMINDER_WAITING)
   {
    nextReminder=r;
    break;
   }
}
int add2RemindeList (Reminder *r)
{
 reminderList.list[reminderList.anz] =r;
 reminderList.anz ++;
 if(reminderList.anz > reminderList.max)
  extendReminderList();
}
int updateReminderList ()
{
 // FIXME sort RemindeList
 qsort(reminderList.list,reminderList.anz,sizeof(Reminder *),compReminder);
}

int compReminder(const void * p1,const void *p2)
{
 int t1,t2;
 Reminder *r1;

 t1 = (*((Reminder **)p1))->ct.ctrl->tim;
 t2 = (*((Reminder **)p2))->ct.ctrl->tim;
 return(t1-t2);
}

char *date2Str(int datum, int tim,char *t)
{
 int day,mon,year,hour,min;
 
 year = datum /10000;
 mon  = (datum%10000) /100;
 day  = datum %100;
 hour = tim /100;
 min  = tim %100;
 
 sprintf(t,"%2d.%2d.%4d, %2d:%02d Uhr",day,mon,year,hour,min);
 return(t);
}
char *date2Str(int datum, char *t)
{
 int day,mon,year,hour,min;
 
 year = datum /10000;
 mon  = (datum%10000) /100;
 day  = datum %100;
 
 sprintf(t,"%2d.%2d.%4d",day,mon,year);
 return(t);
}
char *time2Str(int tim,char *t)
{
 int day,mon,year,hour,min;
 
 hour = tim /100;
 min  = tim %100;
 
 sprintf(t,"%2d:%02d Uhr",hour,min);
 return(t);
}

int weekDayByDate(int datum)
{
 struct tm tm,*tmp;
 int tim;

 tm.tm_mday = datum %100;
 tm.tm_mon  = (datum%10000)/100 -1;
 tm.tm_year = datum /10000-1900;
 tm.tm_hour = 12;
 tm.tm_min  =tm.tm_sec =0;
 tmp=&tm;
 tim = mktime(&tm);
 tmp =localtime((const time_t *)&tim);
 return((tmp->tm_wday+6) %7);
}

int calcDate (int *datum, int *tim, int offset, int offset2 , int unit)
{
 int day,mon,year,hour,min;
 int monLen[13]={0,31,28,31,30,31,30,31,31,30,31,30,31},feb;
 struct tm tms;

 // printf("shift %d, %d  by %d/%d %c to \t",*datum,*tim,offset,offset2,unit);

 year = *datum /10000;
 mon  = (*datum%10000) /100;
 day  = *datum %100;
 hour = *tim /100;
 min  = *tim %100;
 
 switch (unit)
  {
  case 'm': 
   min  += offset; 
   break;
  case 'h': 
   hour += offset;
   min  += offset2;
   break;
  case 'd':
   day  += offset;
   hour += offset2;
   break;
  case 'w' :
   day  += offset*7 + offset2;
   break;
  case 'M' :
   mon  += offset;
   day  += offset2;
   break;
  case 'y' :
   year += offset;
   mon  += offset2;
   break;

  default:
   return(false);
  }



 while(min < 0)
  {
   min+=60;
   hour  --;
  }
 while(min>60)
  {
   min-=60;
   hour++;
  }
 while(hour<0)
  {
   hour += 24;
   day --;
  }
 while(hour >=24)
  {
   hour -=24;
   day++;
  }
 while(day<0)
  {
   mon--;
   if(mon<0)
    {
     mon=12;
     year--;
    }
   day += monLen[mon];
   if(mon ==2)
    if( (year %4) == 0)
     day ++;
  }
 

 while (mon>12)
  {
   mon-=12;
   year++;
  }

 if(mon==2 && (year %4)==0)
  feb=1;
 else
  feb=0;

 while(day>monLen[mon]+feb)
  {
   day -= (monLen[mon]+feb);
   mon++;
   if(mon==2 && (year %4)==0)
    feb=1;
   else
    feb=0;
   
   if(mon>12)
    {
     year++;
     mon=1;
    }
  }

 *datum = year*10000+mon*100+day;
 *tim   = 100*hour+min;
 return(true);
}

int getWeekDayInMonth(int *weekCount, int *wday,int datum)
{
 struct tm tms,*tmp;
 int tim,day;

 tms.tm_sec =  0;
 tms.tm_min =  0;
 tms.tm_hour= 12;
 tms.tm_mday= day = datum %100;
 tms.tm_mon = (datum%10000) /100 -1;
 tms.tm_year= datum /10000-1900;

 tim = mktime(&tms);
 tmp = localtime((const time_t*) &tim);
 
 *wday = (tmp->tm_wday+6)%7;
 *weekCount = (int)floor( (day-0.5)/7.);
}

int getCountedWeekDayDatum(int weekCount, int wday, int datum)
{
 struct tm tms,*tmp;
 int dw,w1day,day,tim;

 tms.tm_sec =  0;
 tms.tm_min =  0;
 tms.tm_hour= 12;
 tms.tm_mday= 1;
 tms.tm_mon = (datum%10000) /100 -1;
 tms.tm_year= datum /10000-1900;
 tim = mktime(&tms);
 tmp = localtime((const time_t*) &tim);

 w1day = (tmp->tm_wday+6) % 7;
 dw    = (wday -w1day +7) % 7;
 day   = dw + weekCount*7 + 1;   // Die 1 , da der Refere4bnz tag der 1. des Monats ist
 datum = datum -datum %100 + day;
 return(datum);
}

int calcDateBackward (int datum, int tim, int offset , int offset2, int unit)
{
 int day,mon,year,hour,min;
 int monLen[13]={0,31,28,31,30,31,30,31,31,30,31,30,31};
 struct tm tms;
 time_t ntim;

 year = datum /10000;
 mon  = (datum%10000) /100;
 day  = datum %100;
 hour = tim /100;
 min  = tim %100;

//  if(verbose)
//   printf("calcBack: %2d.%2d.%4d %2d:%02d  by %d %d %c\t",day,mon,year,hour,min,offset,offset2,unit);

 switch (unit)
  {
  case 'm': 
   min  -= offset; 
   break;
  case 'h': 
   hour -= offset;
   min  -= offset2;
   break;
  case 'd':
   day  -= offset;
   hour -= offset2;
   break;
  default:
   return(0);
  }

 while(min < 0)
  {
   min+=60;
   hour  --;
  }
 while(hour<0)
  {
   hour += 24;
   day --;
  }
 while(mon<1)
  {
   mon+=12;
   year --;
  }
 while(day<0)
  {
   mon--;
   if(mon<0)
    {
     mon=12;
     year--;
    }
   day += monLen[mon];
   if(mon ==2)
    if( (year %4) == 0)
     day ++;
  }

//  if(verbose)
//   printf("-> %2d.%2d.%4d %2d:%02d  \t",day,mon,year,hour,min);

 tms.tm_sec =  0;
 tms.tm_min =min;
 tms.tm_hour=hour;
 tms.tm_mday=day;
 tms.tm_mon =mon-1;
 tms.tm_year=year-1900;

 ntim =mktime(&tms);
//  if(verbose)
//   printf("-> %d\n",ntim);
 
 return(ntim);
}
int calcDateForward (int datum, int tim, int offset , int offset2, int unit)
{
 int day,mon,year,hour,min;
 int monLen[13]={0,31,28,31,30,31,30,31,31,30,31,30,31};
 struct tm tms;

 year = datum /10000;
 mon  = (datum%10000) /100;
 day  = datum %100;
 hour = tim /100;
 min  = tim %100;

 switch (unit)
  {
  case 'm': 
   min  += offset; 
   break;
  case 'h': 
   hour += offset;
   min  += offset2;
   break;
  case 'd':
   day  += offset;
   hour += offset2;
   break;
  default:
   return(0);
  }

 while(min >60)
  {
   min-=60;
   hour++;
  }
 while(hour>23)
  {
   hour -= 24;
   day ++;
  }
 while(mon>12)
  {
   mon-=12;
   year++;
  }
 while(day>monLen[mon])
  {
   day -= monLen[mon];
   if(mon ==2)
    if( (year %4) == 0)
     day --;
   mon++;
   if(mon>12)
    {
     mon=1;
     year++;
    }
  }

 tms.tm_sec =  0;
 tms.tm_min =min;
 tms.tm_hour=hour;
 tms.tm_mday=day;
 tms.tm_mon =mon-1;
 tms.tm_year=year-1900;

 
 return(mktime(&tms));
}

int check4Reminder()
{
 int r,ctim;

 ctim   = time(NULL) ;
 ctim  -= (ctim %60);

 //FIXME only for debugging:
 nextReminder=0;

 if(verbose)
  printf("Checking Reminders (%d .. %d)\n",nextReminder,reminderList.anz);
 for(r=nextReminder; r<reminderList.anz; r++)
  {
   
   if(verbose)
    {
     Date *d;
     DateShift *dsh;
     char *dstr;

     if( (reminderList.list[r]->typ & REMINDER_FOR_TODO) == 0)
      {
       if( (reminderList.list[r]->typ & REMINDER_SHIFT) > 0)
	{
	 dsh = (DateShift *) reminderList.list[r]->date;
	 d = dateList[dsh->did];
	 dstr="dateShift";
	}
       else
	{
	 d = (Date * ) reminderList.list[r]->date;
	 dstr="Date";
	}
      }
     printf("\tReminder %3d for %s %d/'%s' : stat=%d, via=%d\n\t\t\ttim    = %d =\t%s\t\t\tcurTime= %d, delta =%d\n",
	    r,dstr,d->id,d->descr,reminderList.list[r]->status,reminderList.list[r]->typ,
	    reminderList.list[r]->ct.ctrl->tim,ctime((time_t*)&reminderList.list[r]->ct.ctrl->tim),ctim,
	    ctim-reminderList.list[r]->ct.ctrl->tim);
     printf("\t\t\tlastSend= %d, %s\n",reminderList.list[r]->ct.ctrl->lastSendTim,ctime((time_t*)&reminderList.list[r]->ct.ctrl->lastSendTim));
    }
   if(reminderList.list[r]->status == REMINDER_WAITING)
    {
     if(reminderList.list[r]->ct.ctrl->tim-ctim <30)
      {
       sendReminder(reminderList.list[r]);
      }
    }
  }
 for(r=nextReminder; r<reminderList.anz; r++)
  if(reminderList.list[r]->status == REMINDER_WAITING)
   {
    nextReminder=r;
    break;
   }
 if(fprl!=NULL)
  fflush(fprl);
}

int sendReminder (Reminder *r)
{
 int i,ret;
 Date *d;
 DateShift *dsh;
 int *m2u,j;

 // FIXME Remidne rfor TODOS

 if( (r->typ & REMINDER_FOR_TODO)==0 )
  {
   if( (r->typ & REMINDER_SHIFT) > 0)
    {
     dsh = (DateShift *) r->date;
     d= dateList[dsh->did];
    }
   else
    d= (Date * ) r->date;

   if(fprl!=NULL)
    {
     time_t tim;
     tim=time(NULL);
     fprintf(fprl,"%s: Sending reminder on date  %d\'%s'\n",strChop(ctime(&tim)),d->id,d->descr);
    }
  }

 if(r->id==REMIND_ALL_USER || r->id==REMIND_ALL_INTERN || r->id==REMIND_ALL_EXTERN)
  {
   m2u = new int [userAnz];
   for(i=0;i<=userAnz; i++)
    m2u[i]=0;

   for(i=0;i<d->partAnz;i++)
    {
     if(r->id!=REMIND_ALL_EXTERN)  // mail geht an intern oder alle
      {
       if(d->part[i].id<GROUPOFFSET)
	m2u[d->part[i].id] = 1;
       else if(d->part[i].id<ROOMOFFSET)
	for(j=0;j<groupList[d->part[i].id-GROUPOFFSET].anz; j++)
	 m2u[ groupList[d->part[i].id-GROUPOFFSET].membr[j]] =1;
      } 
     if(r->id!=REMIND_ALL_INTERN) //mail an extern oder alle
      {
       if(d->part[i].id>=ADDRESOFFSET && d->part[i].id<=MAX_ID)
	sendreminder2User(r,d->part[i].id);
      }
    }
   for(i=0;i<=userAnz; i++)
    if(m2u[i]==1)
     sendreminder2User(r,i);
   delete m2u;
  }
 else if(r->id<GROUPOFFSET || (r->id >=ADDRESOFFSET && r->id<=MAX_ID))  
  sendreminder2User(r,r->id);
 else
  sendreminder2Group(r,r->id-GROUPOFFSET);
 
 r->ct.ctrl->lastSendTim=r->ct.ctrl->tim;
 d->dirty =true;
 r->status=REMINDER_DONE;
 return(true);
}

int sendreminder2User(Reminder *r,int id)
{
 int ret, direction;

 if(id <GROUPOFFSET)
  {
   switch (r->typ & REMINDER_TYPE_MASK)
    {
    case REMINDER_EINLADUNG: direction= userList[id].direction_dateInvite; break;
    case REMINDER_ERINNERUNG: 
     {
      if( (r->typ & REMINDER_FOR_TODO) >0)
       direction= userList[id].direction_dateRemind;
      else
       direction= userList[id].direction_dateRemind;
      break;
     }
    case REMINDER_MAHNUNG:   direction= userList[id].direction_todoMahn; break;
    case REMINDER_SHIFT: direction= REMINDER_ALWAYS_BY_MAIL; break;
    default: direction =0;
    }
  }
 else
  {
   direction = REMINDER_ALWAYS_BY_MAIL;
  }

 if(direction== REMINDER_ALWAYS_BY_MAIL)
  ret=sendReminderByEmail(r,id);
 else if (direction== REMINDER_ALWAYS_BY_DIALOG)
  ret=sendReminderByDialog(r,id);
 else 
  {
   if( (r->typ & REMINDER_BY_EMAIL) >0)
    ret=sendReminderByEmail(r,id);
   else  if( (r->typ & REMINDER_BY_DIALOG) >0)
    ret=sendReminderByDialog(r,id);
  }
 return(ret);
}
int sendreminder2Group(Reminder *r,int id)
{
 int ret,i,uid,direction;
 for(i=0; i<groupList[id].anz; i++)
  {
   uid = groupList[id].membr[i];

   switch (r->typ & REMINDER_TYPE_MASK)
    {
    case REMINDER_EINLADUNG: direction= userList[uid].direction_dateInvite; break;
    case REMINDER_ERINNERUNG: 
     {
      if( (r->typ & REMINDER_FOR_TODO) >0)
       direction= userList[uid].direction_dateRemind;
      else
       direction= userList[uid].direction_dateRemind;
      break;
     }
    case REMINDER_MAHNUNG:   direction= userList[uid].direction_todoMahn; break;
    default: direction =0;
    }


   if(direction== REMINDER_ALWAYS_BY_MAIL)
    ret=sendReminderByEmail(r,uid);
   else if (direction== REMINDER_ALWAYS_BY_DIALOG)
    ret=sendReminderByDialog(r,uid);
   else 
    {
     if( (r->typ & REMINDER_BY_EMAIL) >0)
      ret = sendReminderByEmail(r,uid);
     else  if( (r->typ & REMINDER_BY_DIALOG) >0)
      ret = sendReminderByDialog(r,uid);
    }
  }
 return(ret);
}

int sendReminderByEmail(Reminder *r,int uid)
{
 char cmd[17000],text[16384],mail[128],smtp[128],subj[512];
 char debugInfo[8192],rt[256];
 char start[64],stop[64];
 Date *d;
 DateShift *dsh;
 TodoItem *todo;
 int i,me,me2,j;
 int send2Extern=false;

 if(noReminder)
  return(true);

 d= (Date * ) r->date;
 dsh = (DateShift* )  r->date;
 todo= (TodoItem * ) r->date;
 if( (r->typ & REMINDER_SHIFT) >0)
  d=dateList[dsh->did];

 if( (r->typ & REMINDER_FOR_TODO) >0)
  if(ignoreReminder(d->id, uid))
   {
    if(verbose)
     printf("Skip Reminder on Date %d/%s for User %d/%s\n",d->id,d->descr,uid,userList[uid].name);
    return(true);
   }

 for(i=0;i<4; i++)
  {
   if(uid>=ADDRESOFFSET && uid<=MAX_ID)
    {
     int idx;
     if(i>0)
      continue;
     idx = aid2idx(uid-ADDRESOFFSET);
     if(idx<0)
      continue;
     strcpy(mail,&adrList[idx]->mail[0][0]);
     strcpy(smtp,""); // FIXE ggfs festen SMTP fr externe einfhren
     send2Extern=true;
    }
   else
    {
     strcpy(mail,userList[uid].mail[i]);
     strcpy(smtp,userList[uid].smtp[i]);
    }

   if(mail[0] == '\0')
    continue;

   int etyp = r->typ & REMINDER_TYPE_MASK;
   switch (etyp)
    {//switch
    case REMINDER_EINLADUNG: 
     {
      char *lstr,eList[4096],txt[128];
      int eanz;
      sprintf(eList,"%s",userList[d->init.id].full);
      eanz=1;
      for(j=0;j<d->partAnz; j++)
       if(d->part[j].typ=='E')
	{
	 if(d->part[j].id<GROUPOFFSET)
	  sprintf(txt,", %s",userList[d->part[j].id].full);
	 else if(d->part[j].id<ROOMOFFSET && !send2Extern)
	  sprintf(txt,", Gruppe %s",groupList[d->part[j].id-GROUPOFFSET].name);
	 strcat(eList,txt);
	 eanz++;
	}
      switch (serverLang)
       {
       case MINKO_LANG_DE:
	if(eanz >1)
	 lstr="laden";
	else
	 lstr="ld";
	break;
       case MINKO_LANG_EN:
       default:
	if(eanz >1)
	 lstr="invite";
	else
	 lstr="invites";
	break;
       }

      if (send2Extern)
       {
	formatMail(mailMaskEinlExternSubj,subj,r,send2Extern);
	formatMail(mailMaskEinlExternText,text,r,send2Extern);
       }
      else
       {
	formatMail(mailMaskEinlInternSubj,subj,r,send2Extern);
	formatMail(mailMaskEinlInternText,text,r,send2Extern);
       }
     }
     break;
    case REMINDER_ERINNERUNG: 
     {
      if( (r->typ & REMINDER_FOR_TODO) >0)
       {
	formatMail(mailMaskErinToDoSubj,subj,r,0);
	formatMail(mailMaskErinToDoText,text,r,0);
      }
      else
       {
	if (send2Extern)
	 {
	  formatMail(mailMaskErinExternSubj,subj,r,send2Extern);
	  formatMail(mailMaskErinExternText,text,r,send2Extern);
	 }
	else
	 {
	  formatMail(mailMaskErinInternSubj,subj,r,send2Extern);
	  formatMail(mailMaskErinInternText,text,r,send2Extern);
	 }
       }
     }
     break;
    case REMINDER_MAHNUNG: 
     {
	formatMail(mailMaskMahnToDoSubj,subj,r,0);
	formatMail(mailMaskMahnToDoText,text,r,0);
     }
     break;
    case REMINDER_SHIFT:
     {
      if(dsh->deleted)
       {
	if (send2Extern)
	 {
	  formatMail(mailMaskSuspExternSubj,subj,r,send2Extern);
	  formatMail(mailMaskSuspExternText,text,r,send2Extern);
	 }
	else
	 {
	  formatMail(mailMaskSuspInternSubj,subj,r,send2Extern);
	  formatMail(mailMaskSuspInternText,text,r,send2Extern);
	 }
       }
      else
       {
	if (send2Extern)
	 {
	  formatMail(mailMaskShiftExternSubj,subj,r,send2Extern);
	  formatMail(mailMaskShiftExternText,text,r,send2Extern);
	 }
	else
	 {
	  formatMail(mailMaskShiftInternSubj,subj,r,send2Extern);
	  formatMail(mailMaskShiftInternText,text,r,send2Extern);
	 }
       }
      
      break;
    }

    }
   if(smtp[0] == '\0')
    sprintf(cmd,"echo '%s' |mail -s \"%s\" %s",text,subj,mail);
   else
    {
     sprintf(cmd,"%s %s %s  %s \"%s\" \"%s\"",mail2smtp,smtp,mail,replyAddress,subj,text);
    }
   system(cmd);
   printf("\t\t Reminder sent to %s\n---------------------------------------\n\n\n",mail);
   if(fprl!=NULL)
    {
     time_t tim;
     tim=time(NULL);
     
     if(uid>=ADDRESOFFSET && uid<=MAX_ID)
      fprintf(fprl,"%s: Reminder send for %d\'%s' send to address %d\n",strChop(ctime(&tim)),d->id,d->descr,uid-ADDRESOFFSET);
     else
      fprintf(fprl,"%s: Reminder send for %d\'%s' send to %d/%s\n",strChop(ctime(&tim)),d->id,d->descr,uid,userList[uid].name);
    }
  }
 if(verbose)
  printf("everything send\n");
 return(true);
}
int sendReminderByDialog(Reminder *r,int uid)
{
 Date *d;
 TodoItem *todo;
 char buffer[16000];
 char elist[4096];
 char snd[260];
 char tx[4][64];
 int etyp = r->typ & REMINDER_TYPE_MASK;
 int i;

 if(noReminder)
  return(true);

 if(userList[uid].rsocket<0)
  {
   printf("sending Reminder via e-mail instead of Dialog\n");
   return(sendReminderByEmail(r,uid));
  }
 printf("sending Reminder via Dialog\n");
 
 d= (Date * ) r->date;
 todo= (TodoItem * ) r->date;

 if( (r->typ & REMINDER_FOR_TODO) >0)
  if(ignoreReminder(d->id, uid))
   {
    if(verbose)
     printf("Skip Reminder on Date %d/%s for User %d/%s\n",d->id,d->descr,uid,userList[uid].name);
    return(true);
   }

 if(uid>=ADDRESOFFSET )
  return(true);

 
 if(userList[uid].sig.beep==1)
  strcpy(snd,"..beep");
 else
  strcpy(snd,userList[uid].sig.cmd);
  
 switch (etyp)
  {
  case REMINDER_EINLADUNG: 
   {
    strcpy(elist,"");
    for(i=0; i< d->partAnz; i++)
     if(d->part[i].typ == UID_TYP_INIT || d->part[i].typ == UID_TYP_EINL)
      {
       char txt[256];
       if(d->part[i].id<GROUPOFFSET)
	sprintf(txt,"{%s} ",userList[d->part[i].id].full);
       else if(d->part[i].id<ROOMOFFSET)
	sprintf(txt,"{Gruppe %s} ",groupList[d->part[i].id-GROUPOFFSET].name);
       else
	continue;
       strcat(elist,txt);
      }
    sprintf(buffer,"remind %d {%s} {%s} {%s} {%s} {%s} {%s} {%s} - - {%s}",REMINDER_EINLADUNG,
	    date2Str(d->startDate,tx[0]),time2Str(d->startTime,tx[1]),
	    date2Str(d->stopDate,tx[2]),time2Str(d->stopTime,tx[3]),roomList[d->room].name,d->descr,
	    elist,snd);
   }
   break;
  case REMINDER_ERINNERUNG: 
   {
    sprintf(buffer,"remind %d {%s} {%s} {%s} {%s} {%s} {%s} %d %d %c {%s}",REMINDER_ERINNERUNG,
	    date2Str(d->startDate,tx[0]),time2Str(d->startTime,tx[1]),
	    date2Str(d->stopDate,tx[2]),time2Str(d->stopTime,tx[3]),roomList[d->room].name,d->descr,
	    r->offset,r->offset2,r->unit,snd);
    break;
   }
  }
 printf("sending Reminder to users (%s_ reminder App : \n'%s'\n",userList[uid].name,buffer);
 send2Client(userList[uid].rsocket,buffer);
 return(true);
}
int sendReminderBySignal(Reminder *r,int uid)
{
 return(true);
}


int ignoreReminder(int did,int uid)
{
 int i;

 if(uid>userAnz)
  return(false);

 for(i=0;i<userList[uid].dmodiAnz; i++)
  if(userList[uid].dmodi[i].did == did)
   return(userList[uid].dmodi[i].noReminder);
    
 return(false);
}



// Format anweisungen:
// %E Einladene(r)  (bei todo = %I)
// %I Initiator
// %A Absender (=company)
// %M Progname
// %B Anfang des Termin  (nur bei Date und dateShift)
// %S Ende der Termine (date)  oder Zeiltermin (todo) (not in dateShift)
// %R Raum (nur bei Date)
// %T Termin-/Aufgabenberschrift
// %D Termin-/Aufgabendetails
// %l ld oder laden, in Abhngigkeit der Zahl der einladenen
// %H Datum von Heute
// %N Notiz zum Schift (nur bei DateShift)
// %?T print following Text only if strlen(Details)>0
// %?N print following Text only if strlen(Notiz of shift) >0 (only shift)
// %]  endof conditional text started by %?*
// %% = %
char * formatMail(char *mask,char *text,Reminder *r,int ext)
{
 char *lstr,eList[4096],txt[128],tmp[128];
 int eanz;
 char *t,*s;
 int j;
 Date *d;
 DateShift *dsh;
 TodoItem *todo;
 int doPrint;

 d=(Date *) r->date;
 todo= (TodoItem * ) r->date;
 dsh=NULL;
 if( (r->typ & REMINDER_SHIFT) >0)
  {
   dsh = (DateShift *)  r->date;
   d = dateList[dsh->did];
  }

 if ( (r->typ & REMINDER_FOR_TODO) ==0) 
  {
   sprintf(eList,"%s",userList[d->init.id].full);
   eanz=1;
   for(j=0;j<d->partAnz; j++)
    if(d->part[j].typ=='E')
     {
      if(d->part[j].id<GROUPOFFSET)
       sprintf(txt,", %s",userList[d->part[j].id].full);
      else if(d->part[j].id<ROOMOFFSET && !ext)
       {
	switch (serverLang)
	 {
	 case MINKO_LANG_DE:
	  sprintf(txt,", Gruppe %s",groupList[d->part[j].id-GROUPOFFSET].name);
	  break;
	 case MINKO_LANG_EN:
	 default:
	  sprintf(txt,", Group %s",groupList[d->part[j].id-GROUPOFFSET].name);
	  break;
	 }
       }
      strcat(eList,txt);
      eanz++;
     }
   switch (serverLang)
    {
    case MINKO_LANG_DE:
     if(eanz >1)
      lstr="laden";
     else
      lstr="ld";
     break;
    case MINKO_LANG_EN:
    default:
     if(eanz >1)
      lstr="invite";
     else
      lstr="invites";
     break;
    }    
  }
 else
  {
   eanz=0;
   strcpy(eList,"");
   lstr="";
  }

 t=text;
 s=mask;
 doPrint=true;
 while ( *s != '\0')
  {
   if( *s != '%' )
    {
     if(doPrint)
      *t=*s;
    }
   else
    {
     s++;
     *t='\0';
     if( (r->typ & REMINDER_FOR_TODO) >0)
      {
       switch (*s)
	{
	case 'E' :
	case 'I' :strcat(text,userList[todo->init].full); break;
	case 'A' :strcat(text,company); break;
	case 'M' :strcat(text,progName); break;
	case 'T' :strcat(text,todo->descr); break;
	case 'D' :strcat(text,todo->descr+todo->tlen); break;
	case 'S' :strcat(text,date2Str(todo->startDate  ,tmp)); break;
	case 'H' :strcat(text,date2Str(today  ,tmp)); break;
	case '%' :strcat(text,"%"); break;
	case '?' :
	 {
	  s++;
	  switch(*s)
	   {
	   case 'T' : 
	    if (strlen(d->descr+d->tlen)== 0) 
	     doPrint=false;
	    break;
	   }
	  break;
	 }
	case ']' :	  doPrint=true; break;
	}
      }
     else if ( (r->typ & REMINDER_SHIFT) >0)
      {
       switch (*s)
	{
	case 'E' :strcat(text,eList); break;
	case 'I' :strcat(text,userList[d->init.id].full); break;
	case 'A' :strcat(text,company); break;
	case 'M' :strcat(text,progName); break;
	case 'R' :strcat(text,roomList[d->room].name); break;
	case 'T' :strcat(text,d->descr); break;
	case 'D' :strcat(text,d->descr+d->tlen); break;
	case 'l' :strcat(text,lstr); break;
	case 'B' :strcat(text,date2Str(r->ct.ctrl->startDate,r->ct.ctrl->startTime,tmp)); break;
	case 'H' :strcat(text,date2Str(today  ,tmp)); break;
	case 'N' :strcat(text,dsh->addDetails); break;
	case '%' :strcat(text,"%"); break;
	case '?' :
	 {
	  s++;
	  switch(*s)
	   {
	   case 'T' : 
	    if (strChopLen(d->descr+d->tlen)== 0) 
	     doPrint=false;
	    break;
	   case 'N' : 
	    if (strChopLen(dsh->addDetails)== 0) 
	     doPrint=false;
	    break;
	   }
	  break;
	 }
	case ']' :	  doPrint=true; break;
	}
      }
     else
      {
       switch (*s)
	{
	case 'E' :strcat(text,eList); break;
	case 'I' :strcat(text,userList[d->init.id].full); break;
	case 'A' :strcat(text,company); break;
	case 'M' :strcat(text,progName); break;
	case 'R' :strcat(text,roomList[d->room].name); break;
	case 'T' :strcat(text,d->descr); break;
	case 'D' :strcat(text,d->descr+d->tlen); break;
	case 'l' :strcat(text,lstr); break;
	case 'B' :strcat(text,date2Str(r->ct.ctrl->startDate,r->ct.ctrl->startTime,tmp)); break;
	case 'S' :strcat(text,date2Str(r->ct.ctrl->stopDate ,r->ct.ctrl->stopTime ,tmp)); break;
	case 'H' :strcat(text,date2Str(today  ,tmp)); break;
	case '%' :strcat(text,"%"); break;
	case '?' :
	 {
	  s++;
	  switch(*s)
	   {
	   case 'T' : 
	    if (strChopLen(d->descr+d->tlen)== 0) 
	     doPrint=false;
	    break;
	   }
	  break;
	 }
	case ']' :	  doPrint=true; break;
	}
      }
     while(*t != '\0')
      t++;
     t--;
    }
   t++;
   s++;
  }
 // printf("Formated Mail Text:\n'%s'\n",text);
}
int strChopLen(const char *t)
{
 char *s;
 if(strlen(t)==0)
  return (0);
 s = new char [strlen(t)+2];
 strcpy(s,t);
 return(strlen(strChop(s)));
}
